import express from 'express';
import http from 'http';
import cors from 'cors';
import dotenv from 'dotenv';
import { Server } from 'socket.io';
import messageRoutes from './routes/messageRoutes.js';
import ConnectDB from './config/db.js';
import Message from './models/Message.js';

dotenv.config();
ConnectDB();

const app = express();
const server = http.createServer(app);

// Middlewares
app.use(cors({
  origin: ['https://chatapp.ecowear.com', 'https://ecowear.com']
}));
app.use(express.json());

// Routes
app.use('/api/messages', messageRoutes);

// Socket.IO
const io = new Server(server, {
  cors: {
    origin: ['https://chatapp.ecowear.com'],
    methods: ['GET', 'POST'],
  },
  transports: ['websocket'],
});

io.on('connection', async (socket) => {
  console.log('User connected:', socket.id);

  // Load old messages
  const messages = await Message.find().sort({ createdAt: 1 }).limit(50);
  socket.emit('previousMessages', messages);

  socket.on('sendMessage', async (msg) => {
    try {
      const saved = await Message.create({
        text: msg.text,
        sender: msg.sender,
      });

      io.emit('receiveMessage', saved);
    } catch (err) {
      console.error('Message save error:', err);
    }
  });

  socket.on('disconnect', () => {
    console.log('User disconnected:', socket.id);
  });
});

// Start Server
const PORT = process.env.PORT || 5000;
server.listen(PORT, '0.0.0.0', () => {
  console.log(`Server running on port ${PORT}`);
});
